module net.BurtonRadons.spyl.main;

/** @mainpage spyl - A Simple Pythonic Language
  * @section introduction Introduction
  *
  * Spyl (pronounced "spill", "speel", "spile", or "richard") is a language of
  * the Python type.  Its light weight and high structural expressiveness are
  * designed to cater to special tasks like configuration files.
  *
  * (Warning: This isn't done.)
  *
  * @section basics Basic Syntax
  *
  * Integers are input using the form "1", "0x5E", "0b1011", or "0o714".  Floats
  * are input using the form "1.4", "1e6", or "1.4e3".
  * Arithmetic includes addition ("a + b"), subtraction ("a - b"), multiplication ("a * b"),
  * division ("a / b"), exponentiation ("a ** b"), modulation ("a % b"), and negation ("-a").
  * Integers also have left-shift ("a << b"), right-shift ("a >> b"), unsigned
  * right-shift ("a >>> b"), bitwise and ("a & b"), bitwise or ("a | b"), bitwise exclusive or
  * ("a ^ b"), and bit invert ("~a").  An expression cannot be terminated with ";"; there's
  * no need for it, as Spyl expressions are not potentially ambiguous.  The only thing
  * that can start an expression is an identifier.
  *
  * Array constants are input using the form "[a, b, c]"; its values can be any mixed type.
  * Arrays can be indexed ("a [b]") using zero-based indexing and sliced ("a [b .. c]")
  * using inclusive-end ranges.  So "a [0 .. 2]" refers to 0 and 1 in the array.  Either or
  * both slice parameters can be ignored.  You can assign to indexes or slices to
  * change the array.  Unlike in Python, array slices refer to their parents' data.
  *
  * Array constants are input using the an escaped string, such as "hello, world\n".
  *
  * Variables can be assigned using "a = b".  Variables do not need to be declared
  * before assignment.  Spyl's scoping rules are the opposite of Python's; more on
  * that later.  All arithmetic can be combined with assignment, such as with "a **= b".
  * This is identical to "a = a ** b".
  *
  * You can access an object's parameters through "a.b", which
  * accesses the "b" attribute of "a".
  */

import net.BurtonRadons.spyl.expression;
import net.BurtonRadons.spyl.lexer;
import net.BurtonRadons.spyl.mark;
import net.BurtonRadons.spyl.parser;
import net.BurtonRadons.spyl.scope;
import net.BurtonRadons.spyl.statement;
import net.BurtonRadons.spyl.value;

class PrintSymbols : Scope
{
    this ()
    {
        assign ("print", new FunctionValue (&doPrint, "print (args...) - Print each argument converted to a string with a space separator in between, finishing it with a newline."));
    }
    
    void doPrint (Value [] args)
    {
        for (int c; c < args.length; c ++)
        {
            if (c)
                printf (" ");
            printf ("%.*s", args [c].str ());
        }
        
        printf ("\n");
    }
}

class ValueSymbols : Scope
{
    this ()
    {
        assign ("Float", FloatType.create ());
        assign ("Integer", IntegerType.create ());
    }
}

version (spylTest)
{
    char [] test = '
        x = y = 4;
        z = "foo";
        z = z + "bar";
        y = 2.3;
        q = [1, 2, 3] ~ [4, 5, 6];
        r = q [2 .. 4];
        q ~= 7;
        print ("q =", q);
        print ("r =", r);
        print (z [2 .. ]);
    ';
    
    
    int main ()
    {
        Parser parser;
        BlockStatement code;
        
        parser = new Parser ("<<string>>", test);
        code = parser.parseModule ();
        if (code === null)
            return 1;
        
        Scope scope = new Scope;
       
        scope.add (new ValueSymbols);
        scope.add (new PrintSymbols);
        
        code.print ();
        code.evaluate (scope);
        return 0;
    }
}